<?php
/**
 * Messages API Endpoint
 * Retrieves messages for a parent/user
 */

header('Content-Type: application/json');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/Database.php';

if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    $db = new Database();
    $pdo = $db->connect();

    if (!isset($_GET['user_id']) || empty($_GET['user_id'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'User ID is required'
        ]);
        exit;
    }

    $user_id = filter_var($_GET['user_id'], FILTER_SANITIZE_NUMBER_INT);
    $limit = isset($_GET['limit']) ? filter_var($_GET['limit'], FILTER_SANITIZE_NUMBER_INT) : 20;

    try {
        $query = "SELECT 
                    m.id,
                    m.sender_id,
                    m.receiver_id,
                    m.message,
                    m.is_read,
                    m.created_at,
                    u.full_name as sender_name,
                    u.email as sender_email
                  FROM messages m
                  JOIN users u ON m.sender_id = u.id
                  WHERE m.receiver_id = :user_id
                  ORDER BY m.created_at DESC
                  LIMIT :limit";
        
        $stmt = $pdo->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Mark messages as read
        $update_query = "UPDATE messages SET is_read = 1 
                        WHERE receiver_id = :user_id AND is_read = 0";
        $update_stmt = $pdo->prepare($update_query);
        $update_stmt->bindParam(':user_id', $user_id);
        $update_stmt->execute();

        echo json_encode([
            'success' => true,
            'data' => $messages
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
} elseif ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Send a message
    $db = new Database();
    $pdo = $db->connect();

    $input = json_decode(file_get_contents('php://input'), true);

    if (!isset($input['sender_id']) || !isset($input['receiver_id']) || !isset($input['message'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Sender ID, Receiver ID, and message are required'
        ]);
        exit;
    }

    try {
        $query = "INSERT INTO messages (sender_id, receiver_id, message, is_read, created_at)
                  VALUES (:sender_id, :receiver_id, :message, 0, NOW())";
        
        $stmt = $pdo->prepare($query);
        $stmt->bindParam(':sender_id', $input['sender_id']);
        $stmt->bindParam(':receiver_id', $input['receiver_id']);
        $stmt->bindParam(':message', $input['message']);
        
        if ($stmt->execute()) {
            echo json_encode([
                'success' => true,
                'message' => 'Message sent successfully',
                'message_id' => $pdo->lastInsertId()
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Failed to send message'
            ]);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
