<?php
/**
 * Fees API Endpoint
 * Retrieves school fees information for a student
 */

header('Content-Type: application/json');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/Database.php';

if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    $db = new Database();
    $pdo = $db->connect();

    if (isset($_GET['student_id']) && !empty($_GET['student_id'])) {
        $student_id = filter_var($_GET['student_id'], FILTER_SANITIZE_NUMBER_INT);
        
        try {
            $query = "SELECT 
                        id,
                        fee_type,
                        amount_due,
                        amount_paid,
                        balance,
                        due_date,
                        payment_date,
                        payment_method,
                        receipt_number,
                        session_year,
                        term
                      FROM fees
                      WHERE student_id = :student_id
                      ORDER BY session_year DESC, term DESC, due_date DESC";
            
            $stmt = $pdo->prepare($query);
            $stmt->bindParam(':student_id', $student_id);
            $stmt->execute();
            
            $fees = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Calculate totals
            $total_due = 0;
            $total_paid = 0;
            $total_balance = 0;
            
            foreach ($fees as $fee) {
                $total_due += (float)$fee['amount_due'];
                $total_paid += (float)$fee['amount_paid'];
                $total_balance += (float)$fee['balance'];
            }
            
            echo json_encode([
                'success' => true,
                'data' => [
                    'fees' => $fees,
                    'summary' => [
                        'total_due' => $total_due,
                        'total_paid' => $total_paid,
                        'total_balance' => $total_balance
                    ]
                ]
            ]);
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    } else {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Student ID is required'
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
